"use strict";
(() => {
  // src/content/content.ts
  var SIDEBAR_ID = "chatct-sidebar";
  var IFRAME_ID = "chatct-iframe";
  var CASHTAG_REGEX = /\$[A-Za-z][A-Za-z0-9]{1,9}/g;
  var PANEL_WIDTH = 320;
  var PANEL_HEIGHT = 480;
  var HOVER_DELAY = 220;
  var HOVER_SPEED_MAX = 0.8;
  var HOVER_STEADY_MAX = 0.28;
  var HOVER_DECEL_MIN = 0.2;
  var AI_WINDOW_MS = 24 * 60 * 60 * 1e3;
  var HEAT_WINDOW_MS = 30 * 60 * 1e3;
  var HEAT_BUCKET_MS = 60 * 1e3;
  var METRIC_REFRESH_MS = 2e4;
  var panelVisible = false;
  var isDocked = false;
  var hoverTimer = null;
  var lastToken = null;
  var lockX = null;
  var lastAiToken = null;
  var lastAiAt = 0;
  var metricsTimer = null;
  var panelHeight = PANEL_HEIGHT;
  var sessionToken = null;
  var iframeReady = false;
  var hoverIntentToken = null;
  var hoverIntentStartedAt = 0;
  var hoverPointer = { speed: 0, prevSpeed: 0, ts: 0, x: 0, y: 0 };
  var pendingMessages = [];
  var tweetStore = /* @__PURE__ */ new Map();
  var isRuntimeValid = () => {
    try {
      if (!chrome?.runtime?.id) return false;
      chrome.runtime.getManifest();
      return true;
    } catch {
      return false;
    }
  };
  var getExtensionOrigin = () => {
    try {
      return new URL(chrome.runtime.getURL("")).origin;
    } catch {
      return "";
    }
  };
  var isExtensionFrame = (iframe) => {
    if (!iframe) return false;
    const origin = getExtensionOrigin();
    const src = iframe.getAttribute("src") || "";
    return Boolean(origin && src.startsWith(origin));
  };
  var recoverPanelAfterSessionReady = () => {
    if (!sessionToken) return;
    if (lastToken) {
      sendTokenToPanel(lastToken);
      refreshMetrics(lastToken);
    }
    const handle = getCurrentHandle();
    if (handle) {
      sendUserToPanel(handle);
      safeStorageSet({ chatctUser: handle });
    }
  };
  var initSessionToken = () => {
    if (!chrome?.storage?.local || !isRuntimeValid()) return;
    chrome.storage.local.get("chatctSessionToken", (data) => {
      const existing = data.chatctSessionToken;
      if (existing) {
        sessionToken = existing;
        recoverPanelAfterSessionReady();
        return;
      }
      const token = crypto.randomUUID ? crypto.randomUUID() : Math.random().toString(36).slice(2);
      sessionToken = token;
      chrome.storage.local.set({ chatctSessionToken: token });
      recoverPanelAfterSessionReady();
    });
  };
  initSessionToken();
  if (chrome?.storage?.onChanged) {
    chrome.storage.onChanged.addListener((changes, area) => {
      if (area !== "local") return;
      if (!changes.chatctSessionToken) return;
      sessionToken = changes.chatctSessionToken.newValue || null;
      recoverPanelAfterSessionReady();
    });
  }
  var getRuntimeUrl = (path) => {
    try {
      if (!isRuntimeValid()) return null;
      return chrome.runtime.getURL(path);
    } catch {
      return null;
    }
  };
  var createSidebar = () => {
    const existing = document.getElementById(SIDEBAR_ID);
    if (existing) return existing;
    const panelUrl = getRuntimeUrl("panel/panel.html");
    if (!panelUrl) return null;
    const host = document.createElement("div");
    host.id = SIDEBAR_ID;
    host.style.position = "fixed";
    host.style.top = "0";
    host.style.left = "0";
    host.style.width = "0";
    host.style.height = "0";
    host.style.zIndex = "2147483647";
    host.style.background = "transparent";
    const shadow = host.attachShadow({ mode: "open" });
    const style = document.createElement("style");
    style.textContent = `
    :host {
      all: initial;
    }
    .wrapper {
      position: fixed;
      width: ${PANEL_WIDTH}px;
      height: ${PANEL_HEIGHT}px;
      max-height: ${PANEL_HEIGHT}px;
      border-radius: 32px;
      border: 1px solid rgba(90, 98, 120, 0.35);
      box-shadow: 0 16px 36px rgba(0, 0, 0, 0.45);
      background:
        radial-gradient(circle at 20% -15%, rgba(88, 96, 255, 0.18), transparent 55%),
        linear-gradient(180deg, #0a1128 0%, #02040a 100%);
      backdrop-filter: blur(40px);
      -webkit-backdrop-filter: blur(40px);
      overflow: hidden;
      display: none;
      transition: width 160ms ease, height 160ms ease, right 160ms ease, top 160ms ease;
    }
    .wrapper.docked {
      width: ${PANEL_WIDTH}px;
      height: 100vh;
      top: 0;
      right: 0;
      left: auto;
      border-radius: 0;
      border-left: 1px solid rgba(255, 255, 255, 0.08);
    }
    iframe {
      width: 100%;
      height: 100%;
      border: none;
      display: block;
    }
  `;
    const wrapper = document.createElement("div");
    wrapper.className = "wrapper";
    const iframe = document.createElement("iframe");
    iframe.id = IFRAME_ID;
    iframe.src = panelUrl;
    iframe.addEventListener("load", () => {
      iframeReady = true;
      if (lastToken) sendTokenToPanel(lastToken);
      while (pendingMessages.length) {
        const item = pendingMessages.shift();
        if (!item) break;
        postToPanel(item.type, item.payload);
      }
    });
    wrapper.appendChild(iframe);
    shadow.appendChild(style);
    shadow.appendChild(wrapper);
    document.body.appendChild(host);
    return host;
  };
  var extractCashtag = (text) => {
    const matches = text.match(CASHTAG_REGEX);
    if (!matches || matches.length === 0) return null;
    return matches[0].toUpperCase();
  };
  var decodeHref = (href) => {
    try {
      return decodeURIComponent(href);
    } catch {
      return href;
    }
  };
  var getCashtagFromElement = (element) => {
    const textTag = extractCashtag(element.textContent?.trim() || "");
    if (textTag) return textTag;
    if (element instanceof HTMLAnchorElement) {
      const href = element.getAttribute("href") || "";
      const decoded = decodeHref(href);
      const hrefTag = extractCashtag(decoded);
      if (hrefTag) return hrefTag;
    }
    return null;
  };
  var getCashtagFromTarget = (target) => {
    if (!target) return null;
    const anchor = target.closest("a");
    if (anchor) {
      const anchorTag = getCashtagFromElement(anchor);
      if (anchorTag) return anchorTag;
    }
    const isLeafTextElement = target.childElementCount === 0;
    const directTag = isLeafTextElement ? getCashtagFromElement(target) : null;
    if (directTag) return directTag;
    const tweetText = target.closest("[data-testid='tweetText']");
    if (tweetText) {
      let cursor = target;
      while (cursor && cursor !== tweetText) {
        const snippet = (cursor.textContent || "").trim();
        if (snippet && snippet.length <= 24) {
          const tag = extractCashtag(snippet);
          if (tag) return tag;
        }
        cursor = cursor.parentElement;
      }
    }
    return null;
  };
  var postToPanel = (type, payload) => {
    const iframe = document.getElementById(SIDEBAR_ID)?.shadowRoot?.getElementById(IFRAME_ID);
    if (!iframe?.contentWindow || !sessionToken || !isExtensionFrame(iframe)) return;
    iframe.contentWindow.postMessage(
      { type, sessionToken, ...payload },
      getExtensionOrigin()
    );
  };
  var sendTokenToPanel = (token, contextText = "") => {
    if (!sessionToken) return;
    if (!iframeReady) {
      pendingMessages.push({ type: "chatct:token", payload: { token, contextText } });
      return;
    }
    postToPanel("chatct:token", { token, contextText });
  };
  var sendUserToPanel = (handle) => {
    if (!sessionToken) return;
    if (!iframeReady) {
      pendingMessages.push({ type: "chatct:user", payload: { handle } });
      return;
    }
    postToPanel("chatct:user", { handle });
  };
  var sendAiToPanel = (payload) => {
    if (!sessionToken) return;
    if (!iframeReady) {
      pendingMessages.push({ type: "chatct:ai", payload });
      return;
    }
    postToPanel("chatct:ai", payload);
  };
  var sendAiErrorToPanel = (error) => {
    if (!sessionToken) return;
    if (!iframeReady) {
      pendingMessages.push({ type: "chatct:ai-error", payload: { error } });
      return;
    }
    postToPanel("chatct:ai-error", { error });
  };
  var safeSendMessage = (message, callback) => {
    try {
      if (!isRuntimeValid()) return;
      if (callback) {
        chrome.runtime.sendMessage(message, callback);
        return;
      }
      const maybePromise = chrome.runtime.sendMessage(message);
      if (maybePromise && typeof maybePromise.catch === "function") {
        void maybePromise.catch(() => void 0);
      }
    } catch {
      return;
    }
  };
  var isMessageResponse = (value) => {
    return typeof value === "object" && value !== null;
  };
  var safeStorageSet = (payload) => {
    try {
      if (!chrome?.storage?.local || !isRuntimeValid()) return;
      chrome.storage.local.set(payload);
    } catch {
      return;
    }
  };
  var getCurrentHandle = () => {
    const profile = document.querySelector("a[data-testid='AppTabBar_Profile_Link']");
    const href = profile?.getAttribute("href") || "";
    if (href.startsWith("/")) {
      const handle = href.split("/").filter(Boolean)[0];
      if (handle) return `@${handle}`;
    }
    return null;
  };
  var sendHeatToPanel = (payload) => {
    if (!sessionToken) return;
    if (!iframeReady) {
      pendingMessages.push({ type: "chatct:heat", payload });
      return;
    }
    postToPanel("chatct:heat", payload);
  };
  var normalizeText = (text) => text.replace(/\s+/g, " ").trim();
  var extractHandle = (article) => {
    const userName = article.querySelector("[data-testid='User-Name']");
    if (!userName) return "";
    const spans = Array.from(userName.querySelectorAll("span"));
    const handle = spans.map((span) => span.textContent || "").find((txt) => txt.startsWith("@"));
    return handle || "";
  };
  var isLikelyBot = (text, handle) => {
    const lower = text.toLowerCase();
    const handleLower = handle.toLowerCase();
    if (handleLower.includes("bot") || handleLower.includes("airdrop")) return true;
    if (lower.includes("airdrop") || lower.includes("free mint")) return true;
    const tickerCount = (text.match(/\$[A-Za-z][A-Za-z0-9]{1,9}/g) || []).length;
    if (tickerCount >= 4) return true;
    return false;
  };
  var recordTweets = (token) => {
    const tokenUpper = token.toUpperCase();
    const articles = Array.from(document.querySelectorAll("article"));
    const store = tweetStore.get(tokenUpper) || /* @__PURE__ */ new Map();
    const cutoff = Date.now() - AI_WINDOW_MS;
    for (const [key, sample] of store) {
      if (sample.ts < cutoff) store.delete(key);
    }
    for (const article of articles) {
      const textNode = article.querySelector("[data-testid='tweetText']");
      if (!textNode) continue;
      const text = normalizeText(textNode.innerText || textNode.textContent || "");
      if (!text) continue;
      if (!text.toUpperCase().includes(tokenUpper)) continue;
      const handle = extractHandle(article);
      const key = `${handle}|${text}`;
      if (store.has(key)) continue;
      store.set(key, {
        text,
        handle,
        ts: Date.now(),
        isBot: isLikelyBot(text, handle)
      });
    }
    tweetStore.set(tokenUpper, store);
  };
  var getRecentTweets = (token) => {
    const tokenUpper = token.toUpperCase();
    const store = tweetStore.get(tokenUpper);
    if (!store) return [];
    const cutoff = Date.now() - AI_WINDOW_MS;
    return Array.from(store.values()).filter((sample) => sample.ts >= cutoff).slice(-60);
  };
  var computeHeat = (token) => {
    const samples = getRecentTweets(token);
    const now = Date.now();
    const start = now - HEAT_WINDOW_MS;
    const buckets = Math.ceil(HEAT_WINDOW_MS / HEAT_BUCKET_MS);
    const realSeries = Array.from({ length: buckets }, () => 0);
    const botSeries = Array.from({ length: buckets }, () => 0);
    for (const sample of samples) {
      const idx = Math.floor((sample.ts - start) / HEAT_BUCKET_MS);
      if (idx < 0 || idx >= buckets) continue;
      if (sample.isBot) botSeries[idx] += 1;
      else realSeries[idx] += 1;
    }
    const totalReal = realSeries.reduce((sum, val) => sum + val, 0);
    const totalBot = botSeries.reduce((sum, val) => sum + val, 0);
    const total = totalReal + totalBot || 1;
    const realRatio = totalReal / total;
    const botRatio = totalBot / total;
    const slice = (arr, startIdx, length) => arr.slice(startIdx, startIdx + length).reduce((sum, val) => sum + val, 0);
    const recent = slice(realSeries, Math.max(0, buckets - 5), 5) + slice(botSeries, Math.max(0, buckets - 5), 5);
    const previous = slice(realSeries, Math.max(0, buckets - 10), 5) + slice(botSeries, Math.max(0, buckets - 10), 5);
    const heatIndex = previous > 0 ? recent / previous : recent > 0 ? 2 : 1;
    const realSlope = slice(realSeries, Math.max(0, buckets - 5), 5) - slice(realSeries, Math.max(0, buckets - 10), 5);
    const botSlope = slice(botSeries, Math.max(0, buckets - 5), 5) - slice(botSeries, Math.max(0, buckets - 10), 5);
    const signal = realSlope > botSlope ? "Real momentum" : botSlope > realSlope ? "Bot-driven" : "Neutral";
    return {
      heatIndex,
      realRatio,
      botRatio,
      series: { real: realSeries, bot: botSeries },
      signal
    };
  };
  var requestAiSummary = (token) => {
    const now = Date.now();
    if (lastAiToken === token && now - lastAiAt < 20 * 1e3) return;
    lastAiToken = token;
    lastAiAt = now;
    safeSendMessage(
      {
        type: "chatct:request-ai-summary",
        token,
        cacheOnly: false
      },
      (response) => {
        const runtimeError = chrome.runtime?.lastError?.message;
        if (runtimeError) {
          chrome.storage.local.get("chatctLastAi", (data) => {
            const cached = data.chatctLastAi;
            if (cached && String(cached.token || "").toUpperCase() === String(token || "").toUpperCase()) {
              sendAiToPanel(cached);
              return;
            }
            const error = `Extension channel unavailable: ${runtimeError}`;
            sendAiErrorToPanel(error);
            safeStorageSet({ chatctAiError: { token, error, at: Date.now(), source: "content" } });
          });
          return;
        }
        if (!isMessageResponse(response) || !response.ok) {
          const error = String(isMessageResponse(response) ? response.error || "AI temporarily unavailable" : "AI temporarily unavailable");
          sendAiErrorToPanel(error);
          safeStorageSet({ chatctAiError: { token, error, at: Date.now(), source: "content" } });
          return;
        }
        if (response.result) {
          const result = response.result;
          sendAiToPanel(result);
          safeStorageSet({ chatctLastAi: { token, ...result } });
          safeStorageSet({ chatctAiError: null });
          return;
        }
        sendAiToPanel({
          summary: "Fetching social intel...",
          vibeScore: 50,
          vibeLabel: "Neutral",
          pending: true
        });
      }
    );
  };
  var refreshMetrics = (token) => {
    recordTweets(token);
    const metrics = computeHeat(token);
    const payload = { token, ...metrics };
    sendHeatToPanel(payload);
    safeStorageSet({ chatctLastHeat: payload });
    requestAiSummary(token);
  };
  var startMetricsLoop = () => {
    if (metricsTimer) return;
    metricsTimer = window.setInterval(() => {
      if (!lastToken) return;
      refreshMetrics(lastToken);
    }, METRIC_REFRESH_MS);
  };
  var stopMetricsLoop = () => {
    if (!metricsTimer) return;
    window.clearInterval(metricsTimer);
    metricsTimer = null;
  };
  var positionPanel = (x, y) => {
    const wrapper = document.getElementById(SIDEBAR_ID)?.shadowRoot?.querySelector(".wrapper");
    if (!wrapper) return;
    if (isDocked) return;
    const desiredLeft = x + 16;
    const desiredTop = y + 16;
    const left = Math.min(desiredLeft, window.innerWidth - PANEL_WIDTH - 12);
    const top = Math.min(desiredTop, window.innerHeight - panelHeight - 12);
    wrapper.style.left = `${Math.max(left, 12)}px`;
    wrapper.style.top = `${Math.max(top, 12)}px`;
  };
  var showPanel = (token, x, y, contextText = "") => {
    const host = createSidebar();
    if (!host) return;
    const wrapper = host.shadowRoot?.querySelector(".wrapper");
    if (!wrapper) return;
    if (token !== lastToken) {
      sendTokenToPanel(token, contextText);
      safeStorageSet({ chatctLastToken: token });
      lastToken = token;
    }
    safeStorageSet({
      chatctLastTokenContext: {
        token,
        text: contextText,
        at: Date.now()
      }
    });
    const handle = getCurrentHandle();
    if (handle) {
      sendUserToPanel(handle);
      safeStorageSet({ chatctUser: handle });
    }
    recordTweets(token);
    refreshMetrics(token);
    wrapper.classList.remove("docked");
    isDocked = false;
    positionPanel(x, y);
    lockX = x;
    wrapper.style.display = "block";
    panelVisible = true;
    startMetricsLoop();
  };
  var hidePanel = () => {
    const wrapper = document.getElementById(SIDEBAR_ID)?.shadowRoot?.querySelector(".wrapper");
    if (!wrapper) return;
    wrapper.style.display = "none";
    wrapper.classList.remove("docked");
    panelVisible = false;
    isDocked = false;
    lockX = null;
    stopMetricsLoop();
  };
  var updatePointerVelocity = (x, y) => {
    const now = Date.now();
    if (!hoverPointer.ts) {
      hoverPointer = { speed: 0, prevSpeed: 0, ts: now, x, y };
      return;
    }
    const dt = Math.max(1, now - hoverPointer.ts);
    const dx = x - hoverPointer.x;
    const dy = y - hoverPointer.y;
    const distance = Math.sqrt(dx * dx + dy * dy);
    const nextSpeed = distance / dt;
    hoverPointer = {
      speed: nextSpeed,
      prevSpeed: hoverPointer.speed,
      ts: now,
      x,
      y
    };
  };
  var hasHoverIntent = (token) => {
    if (hoverIntentToken !== token || !hoverIntentStartedAt) return false;
    const dwellOk = Date.now() - hoverIntentStartedAt >= HOVER_DELAY;
    if (!dwellOk) return false;
    const speed = hoverPointer.speed;
    const prevSpeed = hoverPointer.prevSpeed;
    const speedOk = speed <= HOVER_SPEED_MAX;
    const decel = prevSpeed > 0 ? (prevSpeed - speed) / prevSpeed : 0;
    const decelOk = decel >= HOVER_DECEL_MIN || speed <= HOVER_STEADY_MAX;
    return speedOk && decelOk;
  };
  document.addEventListener(
    "mousemove",
    (event) => {
      const { clientX, clientY } = event;
      updatePointerVelocity(clientX, clientY);
    },
    true
  );
  document.addEventListener(
    "mouseover",
    (event) => {
      const target = event.target;
      const cashtag = getCashtagFromTarget(target);
      if (!cashtag) return;
      const { clientX, clientY } = event;
      updatePointerVelocity(clientX, clientY);
      if (hoverIntentToken !== cashtag) {
        hoverIntentToken = cashtag;
        hoverIntentStartedAt = Date.now();
      }
      if (panelVisible && !isDocked && lastToken === cashtag) return;
      if (hoverTimer) window.clearTimeout(hoverTimer);
      const tweetTextNode = target?.closest("[data-testid='tweetText']");
      const contextText = normalizeText(
        tweetTextNode?.innerText || tweetTextNode?.textContent || ""
      );
      hoverTimer = window.setTimeout(() => {
        if (!hasHoverIntent(cashtag)) return;
        showPanel(cashtag, clientX, clientY, contextText);
      }, HOVER_DELAY);
    },
    true
  );
  document.addEventListener(
    "mousedown",
    (event) => {
      if (!panelVisible) return;
      const host = document.getElementById(SIDEBAR_ID);
      const path = event.composedPath();
      if (host && path.includes(host)) return;
      hidePanel();
    },
    true
  );
  document.addEventListener(
    "click",
    (event) => {
      const target = event.target;
      const cashtag = getCashtagFromTarget(target);
      if (cashtag) {
        event.preventDefault();
        event.stopPropagation();
        return;
      }
    },
    true
  );
  window.addEventListener("message", (event) => {
    if (!event.data || typeof event.data !== "object") return;
    if (event.origin !== getExtensionOrigin()) return;
    if (!sessionToken || event.data.sessionToken !== sessionToken) return;
    const type = event.data.type;
    if (type === "chatct:resize") {
      const height = event.data.height;
      if (typeof height === "number" && Number.isFinite(height)) {
        panelHeight = Math.min(PANEL_HEIGHT, Math.max(240, Math.min(height, window.innerHeight - 24)));
        const wrapper = document.getElementById(SIDEBAR_ID)?.shadowRoot?.querySelector(".wrapper");
        if (wrapper && !isDocked) {
          wrapper.style.height = `${panelHeight}px`;
        }
      }
      return;
    }
    if (type === "chatct:close") {
      hidePanel();
      return;
    }
    if (type === "chatct:dock") {
      const wrapper = document.getElementById(SIDEBAR_ID)?.shadowRoot?.querySelector(".wrapper");
      if (!wrapper) return;
      wrapper.classList.add("docked");
      wrapper.style.display = "block";
      isDocked = true;
      panelVisible = true;
    }
  });
})();
