// src/background/background.ts
var DEFAULT_SUPABASE_URL = "https://qpkhvgfehuhlmkdrjcmz.supabase.co";
var DEFAULT_SUPABASE_ANON_KEY = "eyJhbGciOiJIUzI1NiIsInR5cCI6IkpXVCJ9.eyJpc3MiOiJzdXBhYmFzZSIsInJlZiI6InFwa2h2Z2ZlaHVobG1rZHJqY216Iiwicm9sZSI6ImFub24iLCJpYXQiOjE3NjkzNTI1ODUsImV4cCI6MjA4NDkyODU4NX0.GQCVc9fqjL3NRjncOpQJYRSVpTUi4cHWG_Ri3ez85Yc";
var normalizeSupabaseUrl = (value) => {
  const text = String(value || "").trim();
  return text && isAllowedSupabaseUrl(text) ? text : DEFAULT_SUPABASE_URL;
};
var normalizeAnonKey = (value) => {
  const text = String(value || "").trim();
  return text || DEFAULT_SUPABASE_ANON_KEY;
};
var normalizeStoredConfig = (base) => ({
  ...base,
  plan: String(base.plan || "free"),
  supabaseUrl: normalizeSupabaseUrl(base.supabaseUrl),
  supabaseAnonKey: normalizeAnonKey(base.supabaseAnonKey),
  openaiApiKey: "",
  geminiApiKey: "",
  grokApiKey: "",
  grokModel: String(base.grokModel || "grok-4-1-fast-non-reasoning"),
  billingCheckoutUrl: String(base.billingCheckoutUrl || "")
});
chrome.runtime.onInstalled.addListener(() => {
  chrome.storage.local.get("chatct", (data) => {
    const existing = data.chatct || {};
    chrome.storage.local.set({
      chatct: normalizeStoredConfig(existing)
    });
  });
});
var parseAiPayload = (content) => {
  try {
    return JSON.parse(content);
  } catch {
    const match = content.match(/\{[\s\S]*\}/);
    if (match) {
      try {
        return JSON.parse(match[0]);
      } catch {
        return null;
      }
    }
    return null;
  }
};
var isAllowedSupabaseUrl = (value) => {
  try {
    const url = new URL(value);
    return url.protocol === "https:" && url.hostname.endsWith(".supabase.co");
  } catch {
    return false;
  }
};
var getFunctionsBase = (supabaseUrl) => {
  try {
    const url = new URL(supabaseUrl);
    if (!isAllowedSupabaseUrl(supabaseUrl)) return null;
    const host = url.hostname.replace(".supabase.co", ".functions.supabase.co");
    return `${url.protocol}//${host}`;
  } catch {
    return null;
  }
};
var sanitizeConfig = (existing, payload) => normalizeStoredConfig({
  ...existing,
  ...payload
});
var openSidePanelForTab = (tabId, token, done) => {
  const openPanel = () => {
    if (!chrome.sidePanel?.open) {
      done({ ok: false, error: "Side panel not supported" });
      return;
    }
    chrome.sidePanel.open({ tabId }, () => {
      if (chrome.runtime.lastError) {
        done({ ok: false, error: chrome.runtime.lastError.message });
        return;
      }
      done({ ok: true });
    });
  };
  if (token) {
    chrome.storage.local.set({ chatctRoomToken: String(token).toUpperCase() });
  }
  openPanel();
};
var AI_GLOBAL_WINDOW_MS = 1e4;
var AI_GLOBAL_WINDOW_LIMIT = 5;
var AI_QUEUE_TICK_MS = 1200;
var AI_TOKEN_REFRESH_COOLDOWN_MS = 15 * 60 * 1e3;
var AI_TOKEN_PENDING_COOLDOWN_MS = 45 * 1e3;
var toAiResultPayload = (token, source, pending) => ({
  token,
  vibeScore: Number(source?.vibe_score ?? source?.vibeScore ?? 50),
  vibeLabel: String(source?.vibe_label ?? source?.vibeLabel ?? "Neutral"),
  sentimentScore: Number(source?.sentiment_score ?? source?.sentimentScore ?? 0),
  confidence: Number(source?.confidence ?? 0),
  volume: Number(source?.volume ?? 0),
  momentum: Number(source?.momentum ?? 0),
  summary: String(source?.summary ?? source?.response ?? source?.analysis ?? "").trim(),
  response: String(source?.response ?? "").trim(),
  pending,
  bullPoints: Array.isArray(source?.bull_points) ? source.bull_points : Array.isArray(source?.bullPoints) ? source.bullPoints : [],
  bearPoints: Array.isArray(source?.bear_points) ? source.bear_points : Array.isArray(source?.bearPoints) ? source.bearPoints : []
});
var getCachedAiResultForToken = async (token) => {
  const data = await chrome.storage.local.get("chatctLastAi");
  const cached = data.chatctLastAi;
  if (!cached) return null;
  if (String(cached.token || "").toUpperCase() !== String(token || "").toUpperCase()) return null;
  const summary = String(cached.summary || cached.response || "").trim();
  if (!summary || /queued for analysis|fetching summary|no shared summary yet|no summary yet/i.test(summary)) return null;
  return { ...cached, pending: false };
};
var buildPendingAiResult = (token) => ({
  token,
  vibeScore: 50,
  vibeLabel: "Neutral",
  sentimentScore: 0,
  confidence: 0,
  volume: 0,
  momentum: 0,
  summary: "Queued for analysis...",
  response: "",
  pending: true,
  bullPoints: [],
  bearPoints: []
});
var isTransientAiError = (message) => /failed to fetch|aborterror|aborted|network|timed out|timeout|empty reply|http2|tls|socket|connection reset|temporarily unavailable/i.test(
  String(message || "").toLowerCase()
);
var aiWindowHits = [];
var aiQueue = [];
var aiQueueBusy = false;
var aiQueueTimer = null;
var aiNextRefreshAllowedByToken = /* @__PURE__ */ new Map();
var aiInflightByToken = /* @__PURE__ */ new Map();
var trimAiWindowHits = () => {
  const now = Date.now();
  aiWindowHits = aiWindowHits.filter((ts) => now - ts < AI_GLOBAL_WINDOW_MS);
};
var canDispatchAiNow = () => {
  trimAiWindowHits();
  return aiWindowHits.length < AI_GLOBAL_WINDOW_LIMIT;
};
var markAiDispatch = () => {
  aiWindowHits.push(Date.now());
  trimAiWindowHits();
};
var canRefreshTokenNow = (token) => {
  const normalized = String(token || "").trim().toUpperCase();
  if (!normalized) return false;
  const nextAllowed = aiNextRefreshAllowedByToken.get(normalized) || 0;
  return Date.now() >= nextAllowed;
};
var markTokenRefreshCooldown = (token, cooldownMs) => {
  const normalized = String(token || "").trim().toUpperCase();
  if (!normalized) return;
  aiNextRefreshAllowedByToken.set(normalized, Date.now() + Math.max(0, cooldownMs));
};
var saveAiError = (token, error) => {
  chrome.storage.local.set({
    chatctAiError: {
      token,
      error,
      at: Date.now(),
      source: "background"
    }
  });
};
var saveAiResult = (result) => {
  chrome.storage.local.set({ chatctLastAi: result, chatctAiError: null });
};
var getAiRuntimeConfig = async () => {
  const data = await chrome.storage.local.get("chatct");
  const config = data.chatct || {};
  const supabaseUrl = normalizeSupabaseUrl(config.supabaseUrl);
  const anonKey = normalizeAnonKey(config.supabaseAnonKey);
  const functionsBase = supabaseUrl ? getFunctionsBase(supabaseUrl) : null;
  if (!functionsBase || !anonKey) return null;
  return {
    functionsBase,
    anonKey,
    grokModel: String(config.grokModel || "grok-4-1-fast")
  };
};
var callAiEndpoint = async (endpoint, token, runtime, cacheOnly) => {
  const bodyPayload = {
    token,
    model: runtime.grokModel
  };
  if (endpoint === "chatct-ai-read") {
    bodyPayload.cache_only = cacheOnly;
  }
  try {
    const res = await fetch(`${runtime.functionsBase}/${endpoint}`, {
      method: "POST",
      headers: {
        Authorization: `Bearer ${runtime.anonKey}`,
        apikey: runtime.anonKey,
        "Content-Type": "application/json"
      },
      body: JSON.stringify(bodyPayload)
    });
    const text = await res.text();
    if (!res.ok) {
      return {
        ok: false,
        status: res.status,
        text,
        data: null,
        transient: res.status === 429 || res.status >= 500
      };
    }
    let data = null;
    try {
      data = text ? JSON.parse(text) : {};
    } catch {
      return {
        ok: false,
        status: 502,
        text: text || "Invalid JSON",
        data: null,
        transient: false
      };
    }
    return {
      ok: true,
      status: res.status,
      text,
      data,
      transient: false
    };
  } catch (error) {
    const text = String(error);
    return {
      ok: false,
      status: 0,
      text,
      data: null,
      transient: isTransientAiError(text)
    };
  }
};
var fetchAiSummary = async (token, runtime, cacheOnly = false) => {
  const normalizedToken = String(token || "").trim().toUpperCase();
  if (!normalizedToken) return { ok: false, error: "Missing token" };
  if (cacheOnly) {
    const readOnly = await callAiEndpoint("chatct-ai-read", normalizedToken, runtime, true);
    if (!readOnly.ok) {
      if (readOnly.transient) {
        const pending = buildPendingAiResult(normalizedToken);
        saveAiResult(pending);
        return { ok: true, result: pending, pending: true };
      }
      return { ok: false, error: readOnly.text || "Cache unavailable" };
    }
    const data2 = readOnly.data;
    if (data2?.pending) {
      const pendingSource = data2?.result || data2;
      const pendingMapped = toAiResultPayload(normalizedToken, pendingSource, true);
      const pendingSummary = String(pendingMapped.summary || pendingMapped.response || "").trim();
      if (pendingSummary) {
        saveAiResult(pendingMapped);
        return { ok: true, result: pendingMapped, pending: true };
      }
      const pending = buildPendingAiResult(normalizedToken);
      saveAiResult(pending);
      return { ok: true, result: pending, pending: true };
    }
    if (!data2) {
      return { ok: false, error: "Cache unavailable" };
    }
    const payload2 = data2?.result || data2;
    const parsed2 = payload2?.summary ? payload2 : parseAiPayload(payload2?.content || "");
    const result2 = {
      token: normalizedToken,
      vibeScore: Number(parsed2?.vibe_score ?? parsed2?.vibeScore ?? 50),
      vibeLabel: String(parsed2?.vibe_label ?? parsed2?.vibeLabel ?? "Neutral"),
      sentimentScore: Number(parsed2?.sentiment_score ?? 0),
      confidence: Number(parsed2?.confidence ?? 0),
      volume: Number(parsed2?.volume ?? 0),
      momentum: Number(parsed2?.momentum ?? 0),
      summary: String(
        parsed2?.summary ?? payload2?.summary ?? parsed2?.response ?? payload2?.response ?? payload2?.analysis ?? ""
      ).trim(),
      response: String(parsed2?.response ?? payload2?.response ?? "").trim(),
      pending: false,
      bullPoints: Array.isArray(parsed2?.bull_points) ? parsed2.bull_points : [],
      bearPoints: Array.isArray(parsed2?.bear_points) ? parsed2.bear_points : []
    };
    saveAiResult(result2);
    return { ok: true, result: result2 };
  }
  fetch(`${runtime.functionsBase}/chatct-room`, {
    method: "POST",
    headers: {
      Authorization: `Bearer ${runtime.anonKey}`,
      apikey: runtime.anonKey,
      "Content-Type": "application/json"
    },
    body: JSON.stringify({ token: normalizedToken })
  }).catch(() => void 0);
  const primary = await callAiEndpoint("chatct-ai-read", normalizedToken, runtime, false);
  if (!primary.ok) {
    if (primary.transient) {
      const pending = buildPendingAiResult(normalizedToken);
      saveAiResult(pending);
      return { ok: true, result: pending, pending: true };
    }
    return { ok: false, error: primary.text || "Grok unavailable" };
  }
  const data = primary.data;
  if (data?.ok === false) {
    if (isTransientAiError(String(data.error || ""))) {
      const pending = buildPendingAiResult(normalizedToken);
      saveAiResult(pending);
      return { ok: true, result: pending, pending: true };
    }
    const error = typeof data.error === "string" && data.error ? data.error : "Grok unavailable";
    return { ok: false, error };
  }
  if (!data) {
    if (primary.transient) {
      const pending = buildPendingAiResult(normalizedToken);
      saveAiResult(pending);
      return { ok: true, result: pending, pending: true };
    }
    return { ok: false, error: "Grok unavailable" };
  }
  if (data?.pending) {
    const pendingSource = data?.result || data;
    const pendingMapped = toAiResultPayload(normalizedToken, pendingSource, true);
    const pendingSummary = String(pendingMapped.summary || pendingMapped.response || "").trim();
    if (pendingSummary) {
      saveAiResult(pendingMapped);
      return { ok: true, result: pendingMapped, pending: true };
    }
    const cacheFallback = await callAiEndpoint("chatct-ai-read", normalizedToken, runtime, true);
    if (cacheFallback.ok && cacheFallback.data?.result) {
      const cachedMapped = toAiResultPayload(
        normalizedToken,
        cacheFallback.data.result,
        Boolean(cacheFallback.data.pending)
      );
      saveAiResult(cachedMapped);
      return { ok: true, result: cachedMapped, pending: Boolean(cacheFallback.data.pending) };
    }
    const pending = buildPendingAiResult(normalizedToken);
    saveAiResult(pending);
    return { ok: true, result: pending, pending: true };
  }
  const payload = data?.result || data;
  const parsed = payload?.summary ? payload : parseAiPayload(payload?.content || "") || payload;
  const result = toAiResultPayload(normalizedToken, parsed, false);
  saveAiResult(result);
  return { ok: true, result };
};
var refreshAiToken = async (token, runtime) => {
  const normalized = String(token || "").trim().toUpperCase();
  if (!normalized) return { ok: false, error: "Missing token" };
  const inflight = aiInflightByToken.get(normalized);
  if (inflight) return inflight;
  if (!canRefreshTokenNow(normalized)) {
    const cached = await getCachedAiResultForToken(normalized);
    if (cached) return { ok: true, result: cached, pending: false, cooldown: true };
  }
  if (!canDispatchAiNow()) {
    const queuePosition = enqueueAiToken(normalized);
    const cached = await getCachedAiResultForToken(normalized);
    if (cached) return { ok: true, queued: true, queuePosition, result: cached, pending: false };
    const pending = buildPendingAiResult(normalized);
    saveAiResult(pending);
    return { ok: true, queued: true, queuePosition, result: pending, pending: true };
  }
  const task = (async () => {
    markAiDispatch();
    const response = await fetchAiSummary(normalized, runtime, false);
    if (response.ok) {
      const isPending = Boolean(response.pending);
      markTokenRefreshCooldown(
        normalized,
        isPending ? AI_TOKEN_PENDING_COOLDOWN_MS : AI_TOKEN_REFRESH_COOLDOWN_MS
      );
    } else {
      markTokenRefreshCooldown(normalized, AI_TOKEN_PENDING_COOLDOWN_MS);
    }
    return response;
  })();
  aiInflightByToken.set(normalized, task);
  try {
    return await task;
  } finally {
    aiInflightByToken.delete(normalized);
  }
};
var processAiQueue = async () => {
  if (aiQueueBusy) return;
  if (!aiQueue.length) {
    if (aiQueueTimer) {
      clearInterval(aiQueueTimer);
      aiQueueTimer = null;
    }
    return;
  }
  if (!canDispatchAiNow()) return;
  aiQueueBusy = true;
  const token = aiQueue.shift();
  if (!token) {
    aiQueueBusy = false;
    return;
  }
  try {
    const runtime = await getAiRuntimeConfig();
    if (!runtime) {
      saveAiError(token, "Supabase functions not configured");
      return;
    }
    const result = await refreshAiToken(token, runtime);
    if (!result.ok) {
      const message = String(result.error || "Grok unavailable");
      if (!isTransientAiError(message)) {
        saveAiError(token, message);
      }
    }
  } catch (error) {
    const message = String(error);
    if (!isTransientAiError(message)) {
      saveAiError(token, message);
    }
  } finally {
    aiQueueBusy = false;
  }
};
var enqueueAiToken = (token) => {
  const normalized = String(token || "").trim().toUpperCase();
  if (!normalized) return 0;
  const existingIndex = aiQueue.indexOf(normalized);
  if (existingIndex >= 0) return existingIndex + 1;
  aiQueue.push(normalized);
  if (!aiQueueTimer) {
    aiQueueTimer = setInterval(() => {
      void processAiQueue();
    }, AI_QUEUE_TICK_MS);
  }
  void processAiQueue();
  return aiQueue.length;
};
chrome.action.onClicked.addListener((tab) => {
  if (!tab.id) return;
  openSidePanelForTab(tab.id, "", () => void 0);
});
chrome.runtime.onMessage.addListener((message, _sender, sendResponse) => {
  if (message?.type === "chatct:get-config" /* GetConfig */) {
    chrome.storage.local.get("chatct", (data) => {
      sendResponse({ ok: true, config: data.chatct || {} });
    });
    return true;
  }
  if (message?.type === "chatct:set-config" /* SetConfig */) {
    chrome.storage.local.get("chatct", (data) => {
      chrome.storage.local.set(
        { chatct: sanitizeConfig(data.chatct || {}, message.payload || {}) },
        () => {
          sendResponse({ ok: true });
        }
      );
    });
    return true;
  }
  if (message?.type === "chatct:open-side-panel" /* OpenSidePanel */) {
    const token = message.token || "";
    const openSidePanel = (tabId2) => {
      openSidePanelForTab(tabId2, token, sendResponse);
    };
    const tabId = message.tabId ?? _sender.tab?.id;
    if (tabId) {
      openSidePanel(tabId);
      return true;
    }
    chrome.tabs.query({ active: true, currentWindow: true }, (tabs) => {
      const activeTabId = tabs[0]?.id;
      if (!activeTabId) {
        sendResponse({ ok: false, error: "No active tab" });
        return;
      }
      openSidePanel(activeTabId);
    });
    return true;
  }
  if (message?.type === "chatct:request-ai-summary" /* RequestAiSummary */) {
    void (async () => {
      const token = String(message.token || "").trim().toUpperCase();
      const cacheOnly = Boolean(message.cacheOnly);
      if (!token) {
        sendResponse({ ok: false, error: "Missing token" });
        return;
      }
      const runtime = await getAiRuntimeConfig();
      if (!runtime) {
        sendResponse({ ok: false, error: "Supabase functions not configured" });
        return;
      }
      try {
        if (cacheOnly) {
          const cachedRead = await fetchAiSummary(token, runtime, true);
          if (!cachedRead.ok) {
            const error = String(cachedRead.error || "Cache unavailable");
            if (isTransientAiError(error)) {
              const pendingResult = buildPendingAiResult(token);
              saveAiResult(pendingResult);
              sendResponse({ ok: true, result: pendingResult, pending: true });
              void refreshAiToken(token, runtime);
              return;
            }
            sendResponse({ ok: false, error });
            return;
          }
          sendResponse({ ok: true, result: cachedRead.result, pending: cachedRead.pending || false });
          if (cachedRead.pending) {
            void refreshAiToken(token, runtime);
          }
          return;
        }
        const result = await refreshAiToken(token, runtime);
        if (!result.ok) {
          const error = String(result.error || "Grok unavailable");
          if (isTransientAiError(error)) {
            const queuePosition = enqueueAiToken(token);
            const cached = await getCachedAiResultForToken(token);
            if (cached) {
              sendResponse({ ok: true, queued: true, queuePosition, result: cached, pending: false });
            } else {
              const pendingResult = buildPendingAiResult(token);
              saveAiResult(pendingResult);
              sendResponse({ ok: true, queued: true, queuePosition, result: pendingResult, pending: true });
            }
            return;
          }
          saveAiError(token, error);
          sendResponse({ ok: false, error });
          return;
        }
        sendResponse({ ok: true, result: result.result, pending: result.pending || false });
      } catch (error) {
        const text = String(error);
        if (isTransientAiError(text)) {
          const queuePosition = enqueueAiToken(token);
          const cached = await getCachedAiResultForToken(token);
          if (cached) {
            sendResponse({ ok: true, queued: true, queuePosition, result: cached, pending: false });
          } else {
            const pendingResult = buildPendingAiResult(token);
            saveAiResult(pendingResult);
            sendResponse({ ok: true, queued: true, queuePosition, result: pendingResult, pending: true });
          }
          return;
        }
        saveAiError(token, text);
        sendResponse({ ok: false, error: text });
      }
    })().catch((error) => {
      sendResponse({ ok: false, error: String(error) });
    });
    return true;
  }
  return false;
});
